myParticleSystem (list rules) {
    integer nprims = llGetNumberOfPrims();
    integer i; for (i=1; i<=nprims; i++)
        llLinkParticleSystem (i, rules);
}

//**********************************************************************************
//                    The Animated Fountain - Head
//**********************************************************************************
// This script triggers the fountain spray, based on commands from the controller
// script

// Fountain Message
integer FOUNTAIN_MSG = 12459;
integer FOUNTAIN_RESET = 12458;

// ID of Head
string sHeadID = " ";

// Command data is received via a link from the controller in a | delimited string
// in the following sequence:
string  sHeads;                 // 0 Which head(s) should act 0=all (string)
float   fSDur;                  // 1 Duration of Scene (float)
float   fHDur;                  // 2 Duration of Head Activation (no activation if 0)
vector  vAccel = <0,0,-3>;      // 3 Particle Acceleration (vector)
                                //   <0,0,0> increases height to about 14m
                                //   This parameter is the major factor for height
float   fAge = 2.0;             // 4 How long the particle is to live (float)
                                //   Larger age causes the drops to fall down longer
                                //   0 1 1 age 4 will cause continual stream
float   fAlphaS = .9;           // 5 Alpha setting from beginning to end of particle's (floats)
float   fAlphaE = .5;           // 6       life
float   fAngleS = .05;          // 7 Angle of stream (float)
float   fAngleE = 0;            // 8 0 0 = narrow, .05 0 = wider, .1 0 wider yet, .1 .1 = like a V
integer iBurst = 90;            // 9 # of particles to generate in a burst (integer)
                                //   Increasing does has no noticable effect
vector  vColorS = <1,1,1>;      // 10 Color setting of particle from beginning to end (vectors)
vector  vColorE = <1,1,1>;      // 11                                                 
integer iGlow = TRUE;           // 12 Whether the emissive flag should be set (TRUE/FALSE)
vector  vOmega = <0,0,6.28>;    // 13 Particle Rotation (vector)
float   fRadius = .5;           // 14 Distance from the emitter for particle creation (float)
float   fSpeedMin = 3.0;        // 15 Minimum initial particle speed (floats)
float   fSpeedMax = 7.0;        // 16                           
vector  vScaleS = <0.1,0.8,0>;  // 17 Particle size at Start/End (vectors)
vector  vScaleE = <0.2,0.3,0>;  // 18                   
float   fSleep = .1;            // 19 Sleep time between bursts (float)
string  sTexture = "";          // 20 Name of particle texture in inventory or UUID (text)

// Particle System Flags
integer iPF_Bounce = 4;
integer iPF_Emissive = 256;
integer iPF_Color = 1;
integer iPF_Scale = 2;
integer iParticleFlags;


execute_particle_system()
{    
    // Set particle Flags
    iParticleFlags = iPF_Bounce + iPF_Color + iPF_Scale;
    if (iGlow)
    {
        iParticleFlags += iPF_Emissive;
    }    
        
    // Execute the system
    myParticleSystem( [
        
        // Particle Flags
        PSYS_PART_FLAGS, iParticleFlags,
            // PSYS_PART_BOUNCE_MASK |
                    // When set, specifies particles will bounce off a plane at the region 
                    // Z height of the emitter. On "bounce", each particle reverses velocity
                    // and angle. This only works for particles above the plane falling down
                    // on it.  0x004
            // PSYS_PART_EMISSIVE_MASK  |
                    // When set, particles are full-bright and are unaffected by global
                    // lighting (sunlight). Otherwise, particles will be lit depending on the
                    // current global lighting conditions. Note that point lights do
                    // illuminate non-emissive particles.  0x100  
            // PSYS_PART_FOLLOW_SRC_MASK |
                    // When set, particles move relative to the position of the emitter. 
                    // Otherwise, particle position and movement are unaffected by the
                    // position/movement of the emitter. This flag disables the
                    // PSYS_SRC_BURST_RADIUS rule.  0x010  
            // PSYS_PART_FOLLOW_VELOCITY_MASK |
                    // When set, particles rotate to orient their "top" towards the direction
                    // of movement or emission. Otherwise, particles are oriented vertically
                    // as their textures would appear (top of texture at top, left at left).
                    //  0x020  
            //PSYS_PART_INTERP_COLOR_MASK |
                    // When set, particle color and alpha transition from their START settings
                    // to their END settings during the particle's lifetime. The transition is
                    // a smooth interpolation.  0x001  
            //PSYS_PART_INTERP_SCALE_MASK |
                    // When set, particle size/scale transitions from its START setting to its
                    // END setting during the particle's lifetime.  0x002  
            // PSYS_PART_TARGET_LINEAR_MASK |
                    // When set, emitted particles move in a straight line towards the target
                    // specified by the PSYS_SRC_TARGET_KEY rule. In this mode,
                    // PSYS_SRC_ACCEL, PSYS_SRC_BURST_RADIUS, and possibly other rules are
                    // ignored.  0x080  
            // PSYS_PART_TARGET_POS_MASK |
                    // When set, emitted particles change course during their lifetime,
                    // attempting to move towards the target specified by the S_SRC_TARGET_KEY
                    // rule by the time they expire. Note that if no target is specified, the
                    // target moves out of range, or an invalid target is specified, the
                    // particles target the prim itself.  0x040  
            // PSYS_PART_WIND_MASK |
                    // When set, particle movement is affected by the wind. It is applied as a
                    // secondary force on the particles.  0x008  
            // 0,
            
        // Particle Patterns
        PSYS_SRC_PATTERN,
            // PSYS_SRC_PATTERN_EXPLODE |
                    // Sprays particles outwards in a spherical area. The Initial velocity of
                    // each particle is determined by PSYS_SRC_BURST_SPEED_MIN and
                    // PSYS_SRC_BURST_SPEED_MAX. The EXPLODE pattern ignores the ANGLE
                    // parameters.  0x02  
               PSYS_SRC_PATTERN_ANGLE_CONE |
                    // Sprays particles outwards in a spherical, sub-spherical, conical or
                    // ring shaped area, as defined by the ANGLE parameters 
                    // PSYS_SRC_ANGLE_BEGIN and PSYS_SRC_ANGLE_END. The ANGLE_CONE pattern
                    // can be used to imitate the EXPLODE pattern by explicitly 
                    // setting PSYS_SRC_ANGLE_BEGIN to 0.00000 and PSYS_SRC_ANGLE_END to
                    // 3.14159 (or PI) (or vice versa).  0x08  
            // PSYS_SRC_PATTERN_ANGLE |
                    // Sprays particles outward in a flat circular, semi-circular, arc or ray
                    // shaped areas, as defined by PSYS_SRC_ANGLE_BEGIN and
                    // PSYS_SRC_ANGLE_END. The circular pattern radiates outwards around the
                    // prim's local X axis line.  0x04  
            // PSYS_SRC_PATTERN_DROP |
                    // Creates particles with no initial velocity. The DROP pattern will
                    // overrides any values given for PSYS_SRC_BURST_RADIUS,                                       
                    // PSYS_SRC_BURST_SPEED_MIN, and PSYS_SRC_BURST_SPEED_MAX, setting each to
                    // 0.00000. (All patterns will behave like the DROP pattern, if RADIUS,
                    // SPEED_MIN and SPEED_MAX are explicitly set to 0.0000.)  0x01 
            0, 
                                            
        // Appearance Settings
        PSYS_SRC_BURST_RADIUS, fRadius,
                    // Specifies the distance from the emitter where particles will be
                    // created. This rule is ignored when the PSYS_PART_FOLLOW_SRC_MASK flag
                    // is set. A test in forums-archive.secondlife.com/327/f5/226722/1.html
                    // indicates that the maximum value is 50.00
        PSYS_SRC_ANGLE_BEGIN, fAngleS,
                    // Specifies a half angle, in radians, of a circular or spherical "dimple" 
                    // or conic section (starting from the emitter facing) within which
                    // particles will NOT be emitted. Valid values are the same as for
                    // PSYS_SRC_ANGLE_END, though the effects are reversed accordingly. If the
                    // pattern is PSYS_SRC_PATTERN_ANGLE, the presentation is a 2D flat
                    // circular section. If PSYS_SRC_PATTERN_ANGLE_CONE or
                    // PSYS_SRC_PATTERN_ANGLE_CONE_EMPTY is used, the presentation is a 3D
                    // spherical section. Note that the value of this parameter and
                    // PSYS_SRC_ANGLE_END are internally re-ordered such that this parameter
                    // gets the smaller of the two values.
        PSYS_SRC_ANGLE_END, fAngleE,
                    // Specifies a half angle, in radians, of a circular or spherical "dimple"
                    // or conic section (starting from the emitter facing) within which
                    // particles will NOT be emitted. Valid values are 0.0, which will result
                    // in particles being emitted in a straight line in the direction of the
                    // emitter facing, to PI, which will result in particles being emitted in
                    // a full circular or spherical arc around the emitter, not including the
                    // "dimple" or conic section defined by PSYS_SRC_ANGLE_BEGIN. If the
                    // pattern is PSYS_SRC_PATTERN_ANGLE, the presentation is a 2D flat
                    // circular section. If PSYS_SRC_PATTERN_ANGLE_CONE or
                    // PSYS_SRC_PATTERN_ANGLE_CONE_EMPTY is used, the presentation is a 3D
                    // spherical section. Note that the value of this parameter and
                    // PSYS_SRC_ANGLE_BEGIN are internally re-ordered such that this parameter
                    // gets the larger of the two values.
        // PSYS_SRC_INNERANGLE,
                    // DEPRECATED: Use PSYS_SRC_ANGLE_BEGIN instead.
        // PSYS_SRC_OUTERANGLE,
                    // DEPRECATED: Use PSYS_SRC_ANGLE_END instead. 
        // PSYS_SRC_TARGET_KEY, NULL_KEY,
                    // Specifies the key of a target object, prim, or agent towards which the
                    // particles will change course and move. They will attempt to end up at
                    // the geometric center of the target at the end of their lifetime. 
                    // Requires the PSYS_PART_TARGET_POS_MASK flag be set.
                                            
        // Particle Appearance  
        PSYS_PART_START_COLOR,  vColorS,
                    // A vector specifying the color of the particles upon emission.
        PSYS_PART_END_COLOR, vColorE,
                    // A vector specifying the color the particles transition to during their
                    // lifetime. Only used if the PSYS_PART_INTERP_COLOR_MASK flag is set.
        PSYS_PART_START_ALPHA, fAlphaS,
                    // Specifies the alpha of the particles upon emission. Valid values are in
                    // the range 0.0 to 1.0. Lower values are more transparent; higher ones
                    // are more opaque.
        PSYS_PART_END_ALPHA, fAlphaE,
                    // Specifies the alpha the particles transition to during their lifetime.
                    // Only used if the PSYS_PART_INTERP_COLOR_MASK flag is set. Valid values
                    // are the same as PSYS_PART_START_ALPHA.
        PSYS_PART_START_SCALE, vScaleS,
                    // Specifies the scale or size of the particles upon emission. Valid
                    // values for each direction are 0.03125 to 4.0, in meters. The actual
                    // particle size is always a multiple of 0.03125. Smaller changes don't
                    // have any effect. Since particles are essentially 2D sprites, the Z
                    // component of the vector is ignored and can be set to 0.0.
        PSYS_PART_END_SCALE, vScaleE,
                    // Specifies the scale or size the particles transition to during their
                    // lifetime. Only used if the PSYS_PART_INTERP_SCALE_MASK flag is set.
                    // Valid values are the same as PSYS_PART_START_SCALE.
        PSYS_SRC_TEXTURE, sTexture,
                    // Specifies the name of a texture in the emitter prim's inventory to use
                    // for each particle. Alternatively, you may specify an asset key UUID for
                    // a texture. If using llLinkParticleSystem and texture is not a UUID,
                    // texture must be in the emitter prim (not necessarily with the script).
                                            
        // Particle Flow  
        PSYS_SRC_MAX_AGE, 0,
                    // Specifies the length of time, in seconds, that the emitter will operate
                    // upon coming into view range (if the particle system is already set) or
                    // upon execution of this function (if already in view range). Upon
                    // expiration, no more particles will be emitted, except as specified
                    // above. Zero will give the particle system an infinite duration.
        PSYS_PART_MAX_AGE, fAge,
                    // Specifies the lifetime of each particle emitted, in seconds. Maximum is
                    // 30.0 seconds. During this time, the particle will appear, change
                    // appearance and move according to the parameters specified in the other
                    // sections, and then disappear.
        PSYS_SRC_BURST_RATE, fSleep,
                    // Specifies the time interval, in seconds, between "bursts" of particles
                    // being emitted. Specifying a value of 0.0 will cause the emission of
                    // particles as fast as the viewer can do so.
        PSYS_SRC_BURST_PART_COUNT, iBurst,
                    // Specifies the number of particles emitted in each "burst".
                                            
        // Particle Motion  
        PSYS_SRC_ACCEL, vAccel,
                    // Specifies a directional acceleration vector applied to each particle as
                    // it is emitted, in meters per second. Valid values are 0.0 to 100.0 for
                    // each direction both positive and negative, as region coordinates.
        PSYS_SRC_OMEGA, vOmega,
                    // Specifies the rotational spin of the emitter in radians per second
                    // along each axis. This "unsticks" the emitter facing from the prim's
                    // positive Z axis and is noticeable in directional presentations. Prim
                    // spin (via llTargetOmega) has no effect on emitter spin.
        PSYS_SRC_BURST_SPEED_MIN, fSpeedMin,
                    // Specifies the minimum value of a random range of values which is
                    // selected for each particle in a burst as its initial speed upon
                    // emission, in meters per second. Note that the value of this parameter
                    // and PSYS_SRC_BURST_SPEED_MAX are internally re-ordered such that this
                    // parameter gets the smaller of the two values.
        PSYS_SRC_BURST_SPEED_MAX, fSpeedMax
                    // Specifies the maximum value of a random range of values which is
                    // selected for each particle in a burst as its initial speed upon
                    // emission, in meters per second. Note that the value of this parameter
                    // and PSYS_SRC_BURST_SPEED_MIN are internally re-ordered such that this
                    // parameter gets the larger of the two values. 
        ]);
} 

default
{
    state_entry()
    {
        // Reset Particle System
        myParticleSystem([]);
        
        // Get Head ID
        sHeadID = llStringTrim(llGetSubString(llGetObjectDesc(), 0, 0), STRING_TRIM);
        
        // Disallow Head 0
        if (sHeadID == "0")
        {
            llSay (0, "Nozzle ID may not be zero, changed to blank");
            sHeadID = " ";
        } 
    }

    on_rez(integer parm)
    {
        llResetScript();
    }
    
    // Process Controller Commands
    link_message(integer source, integer num, string data, key ID)
    {
        if (num == FOUNTAIN_MSG)
        {
            // Parse Command
            list l = llParseStringKeepNulls(data, ["|"], []);
            
            // Test if for this head
            sHeads = llList2String(l, 0);
            integer match_head = FALSE;
            if (sHeads == "0")
            {
                match_head = TRUE;
            }
            else
            {
                integer i;
                for (i = 0; i < llStringLength(sHeads); i++)
                {
                    if (llGetSubString(sHeads, i, i) == sHeadID)
                    {
                        match_head = TRUE;
                    }
                }
            }
            if (match_head)
            {
                
                // Set Timing values
                fSDur = (float)llList2String(l, 1);
                fHDur = (float)llList2String(l, 2);
                
                // Set particle system values
                string s = llList2String(l, 3);
                if (s != "")
                {
                    vAccel = (vector)s;
                }

                s = llList2String(l, 4);
                if (s != "")
                {
                    fAge = (float)s;
                }

                s = llList2String(l, 5);
                if (s != "")
                {
                    fAlphaS = (float)s;
                }
                
                s = llList2String(l, 6);
                if (s != "")
                {
                    fAlphaE = (float)s;
                }
                
                s = llList2String(l, 7);
                if (s != "")
                {
                    fAngleS = (float)s;
                }
                
                s = llList2String(l, 8);
                if (s != "")
                {
                    fAngleE = (float)s;
                }
                
                s = llList2String(l, 9);
                if (s != "")
                {
                    iBurst = (integer)s;
                }
                
                s = llList2String(l, 10);
                if (s != "")
                {
                    vColorS = (vector)s;
                }
                
                s = llList2String(l, 11);
                if (s != "")
                {
                    vColorE = (vector)s;
                }
                
                s = llList2String(l, 12);
                if (s != "")
                {
                    if (s == "Yes")
                    {
                        iGlow = TRUE;
                    }
                    else
                    {
                        iGlow = FALSE;
                    }
                }
                
                s = llList2String(l, 13);
                if (s != "")
                {
                    vOmega = (vector)s;
                }
                
                s = llList2String(l, 14);
                if (s != "")
                {
                    fRadius = (float)s;
                }
                
                s = llList2String(l, 15);
                if (s != "")
                {
                    fSpeedMin = (float)s;
                }
                
                s = llList2String(l, 16);
                if (s != "")
                {
                    fSpeedMax = (float)s;
                }
                
                s = llList2String(l, 17);
                if (s != "")
                {
                    vScaleS = (vector)s;
                }
                
                s = llList2String(l, 18);
                if (s != "")
                {
                    vScaleE = (vector)s;
                }
                
                s = llList2String(l, 19);
                if (s != "")
                {
                    fSleep = (float)s;
                }
                
                s = llList2String(l, 20);
                if (s != "")
                {
                    sTexture = s;
                }
                
                s = llList2String(l, 21);
                if (s != "")
                {
                    // Set new head rotation
                    llSetRot(llEuler2Rot((vector)s * DEG_TO_RAD));
                }
                
                // Test whether activation of head is wanted
                if (fHDur > 0.0)
                {
                
                    // Turn off last spray pattern
                    llSetTimerEvent(0.0);
                    myParticleSystem([]);
                

                    // Display the new spray pattern
                    llSetTimerEvent(fHDur);
                    execute_particle_system();
                }
            }
        }
        
        // Controller Reset
        else if (num == FOUNTAIN_RESET)
        {
            llResetScript();
        }
    }
    
    // Timeout
    timer()
    {
        llSetTimerEvent(0.0);
        myParticleSystem([]);
    }
}
