COMMAND (integer link, integer num, string str, key id) {
    string cmd;
    if (num == FOUNTAIN_MSG)    cmd = "FOUNTAIN_MSG";
    if (num == FOUNTAIN_RESET)  cmd = "FOUNTAIN_RESET";
    if (num == FOUNTAIN_CONFIG) cmd = "FOUNTAIN_CONFIG";

    // link est toujours LINK_SET
    // id est toujours NULL_KEY
    
    //llOwnerSay ("<"+cmd+"> <"+str+">");
    llMessageLinked (LINK_THIS,num,str,id );
}


//**********************************************************************************
//                    The Animated Fountain - Controller
//**********************************************************************************


// Debug Mode
integer DB = FALSE;                             // Set to TRUE to enable debugging
integer iDB = 0;                                // Debug linecount (can get out of seq)

// Message Constants
integer FOUNTAIN_MSG = 12459;
integer FOUNTAIN_RESET = 12458;
integer FOUNTAIN_CONFIG = 12457;
 
 // Notecard info
key kNC;                                        // Key
integer iNC = 0;                                // Current line number
list lNCNames = [];                             // List of Notecards
integer iNCCount;
integer iActNC = 0;                             // Active NC

// Do Stack
list lDOTarget = [];
list lDOSource = [];
list lDOCount = [];

// Commands
integer iCPtr = 0;                              // Command Pointer
list lCommands = [];                            // Command Table

// There are 5 types of entries in the Command list, each headed by a Command ID. Fields within
// each entry are separated by the '|' character.

// Command entry (sCommand = "C"
string   sCommand;               // "C" (this field is not sent to the Heads
string   sHeads;                 // 0 Which head(s) should act 0=all (string)
string   sCDur;                  // 1 Duration of Command (float)
string   sHDur;                  // 2 Duration of Head Activation (0 means just apply parms)
string   sAccel;                 // 3 Particle Acceleration (vector)
string   sAge;                   // 4 How long the particle is to live (float)
string   sAlphaS;                // 5 Alpha setting from beginning to end of particle's (floats)
string   sAlphaE;                // 6       life
string   sAngleS;                // 7 Complex - Read in Particle system description (float)
string   sAngleE;                // 8
string   sBurst;                 // 9 # of particles to generate in a burst (integer)
string   sColorS;                // 10 Color setting of particle from beginning to end (vectors)
string   sColorE;                // 11                                                 
string   sGlow;                  // 12 Whether the emissive flag should be set (TRUE/FALSE)
string   sOmega;                 // 13 Particle Rotation (vector)
string   sRadius;                // 14 Distance from the emitter for particle creation (float)
string   sSpeedMin;              // 15 Minimum initial particle speed (floats)
string   sSpeedMax;              // 16                           
string   sScaleS;                // 17 Particle size at Start/End (vectors)
string   sScaleE;                // 18                   
string   sSleep;                 // 19 Sleep time between bursts (float)
string   sTexture;               // 20 Name of particle texture in inventory or UUID (text)
string   sRotation;              // 21 Rotational Euler degree values

// GROUP entry (sCommand "G")
// This entry gives the name of the Group.
// string sCommand;
// string sGroupName;


// DO entry (sCommand "D")
// string sCommand;
// string sDOTargetName;
// string sDOTargetPtr;
// string sDOCount;

// SAY entry (sCommand "S")
// string sCommand
// string chat_channel
// string message

// MSG entry (sCommand "M")
// string sCommand
// string Linked Message Integer
// string message



// Key / Parameter Validation
list lVKeys = ["ACCEL", "AGE", "ALPHA", "ANGLE", "BURST", "COLOR", "GLOW", "OMEGA", "RADIUS",
                "SPEED", "SCALE", "SLEEP", "TEXTURE", "ROTATION"];
string sVKey;
integer iVParms;
string sP1;
string sP2;
string sP3;
string sP4;
string sP5;
string sP6;

// Fountain Administrators
list lAdmin = [];

// Fountain on/off status
integer iFountainOn = FALSE;

// Menu Information
integer iMenuChan = -9632;
list lMenu = [];
integer iMenu = 0;
key kAvKey;

//***********************************************
// Tests for valid float with/without decimal as text
//***********************************************
integer is_float(string s)
{
    integer sl = llStringLength(s);
    if (sl < 1)
    {
        return FALSE; 
    }
    integer i;
    integer have_dec = FALSE;
    integer have_digit = FALSE;
    for (i = 0; i< sl; i++)
    {
        integer si = llSubStringIndex("0123456789.-", llGetSubString(s, i, i));
        if (si < 0)
        {
            return FALSE;
        }
        if (si == 10)
        {
            if (have_dec)
            {
                return FALSE;
            }
            have_dec = TRUE;
        }
        if ((si == 11) && (i > 0))
        {
            return FALSE;
        }
        have_digit = TRUE;
    }
    if (!have_digit)
    {
        return FALSE;
    }
    return TRUE;
}

//***********************************************
// Tests for valid integer as text
//***********************************************
integer is_integer(string s)
{
    integer sl = llStringLength(s);
    if (sl < 1)
    {
        return FALSE;
    }
    integer i;
    for (i = 0; i< sl; i++)
    {
        integer si = llSubStringIndex("0123456789-", llGetSubString(s, i, i));
        if (si < 0)
        {
            return FALSE;
        }
        if (si == 10)
        {
            if ((i > 0) || (sl == 1))
            {
                return FALSE;
            }
        }
    }
    return TRUE;
}

//******************************************************************
string max_12(list l, integer i) // Return Max 12 char string from list
//******************************************************************
{
    string s = llList2String(l, i);
    
    if (llStringLength(s) > 12)
    {
        return llGetSubString(s, 0, 11);
    }
    return s;
}    

//***********************************************
integer get_keyvalues(list l, integer p)
//***********************************************
{
    // Clear key/values
    sVKey = "";
    iVParms = 0;
    list lp = [];
    
    // Validate key
    if (llListFindList(lVKeys, [llToUpper(llList2String(l, p))]) == -1)
    {
        return FALSE;
    }
    sVKey = llToUpper(llList2String(l, p));
    
    // Get up to 6 parameters
    p++;
    while ((p < llGetListLength(l)) && 
        (llListFindList(lVKeys, [llToUpper(llList2String(l, p))]) == -1) &&
        (llList2String(l, p) != "//") && (iVParms < 6))
    {
        lp += [llList2String(l, p)];
        iVParms++;
        p++;
    }
    
    // Done
    sP1 = llList2String(lp, 0);
    sP2 = llList2String(lp, 1);
    sP3 = llList2String(lp, 2);
    sP4 = llList2String(lp, 3);
    sP5 = llList2String(lp, 4);
    sP6 = llList2String(lp, 5);
    
    return TRUE;
}


//***********************************************
string validation(string s)
//***********************************************
// returns an arror message or the validated string
{
    // Parse command parameters
    s = llStringTrim(s, STRING_TRIM);
    list l = llParseString2List(s, [" "], []);
                
    // Validate at least 3 parameters
    if (llGetListLength(l) < 3)
    {
        return "ENot enough parameters";
    }
                
    // Validate Heads
    sHeads = llList2String(l, 0);
    if (llStringLength(sHeads) < 1)
    {
        return "EInvalid Heads";
    }
                
    // Validate Command Duration
    sCDur = llList2String(l, 1);
    if (!is_float(sCDur))
    {
        return "EInvalid Command Duration";
    }
    
    // Validate Head Duration
    sHDur = llList2String(l, 2);
    if (!is_float(sHDur))
    {
        return "EInvalid Head Duration";
    }

    // Reset Keyed values
    sAccel = "";
    sAge = "";
    sAlphaS = "";
    sAlphaE = "";
    sAngleS = "";
    sAngleE = "";
    sBurst = "";
    sColorS = "";
    sColorE = "";
    sGlow = "";
    sOmega = "";
    sRadius = "";
    sSpeedMin = "";
    sSpeedMax = "";
    sScaleS = "";
    sScaleE = "";
    sSleep = "";
    sTexture = "";
    sRotation = "";
    
    // Validate keyed parms
    integer ptr = 3;
    while ((ptr < llGetListLength(l)) && (llList2String(l, ptr) != "//"))
    {
        // Ensure at least one value
        if ((ptr + 1) >= llGetListLength(l))
        {
            return "ENot enough parameters";
        }
        
        // Capture key and values
        if (!get_keyvalues(l, ptr))
        {
            return "EInvalid Key name";
        }
                    
        // Accel
        if (sVKey == "ACCEL")
        {
            // Ensure three values
            if (iVParms != 3)
            {
                return "EInvalid Accel";
            }
            if ((!is_float(sP1)) || (!is_float(sP2)) || (!is_float(sP3)))
            {
                return "EInvalid Accel";
            }
            sAccel = "<" + sP1 + "," + sP2 + "," + sP3 + ">";
            ptr += 4;
        }
            
        // Age
        else if (sVKey == "AGE")
        {
            if (iVParms != 1)
            {
                return "EInvalid Age";
            }
            if (!is_float(sP1))
            {
                return "EInvalid Age";
            }
            sAge = sP1;
            ptr += 2;
        }
                    
        // Alpha
        else if (sVKey == "ALPHA")
        {
            // Ensure 1-2 values
            if (iVParms > 2)
            {
                return "EInvalid Alpha";
            }
            if (iVParms == 1)
            {
                sP2 = sP1;
            }
            if ((!is_float(sP1)) || (!is_float(sP2)))
            {
                return "EInvalid Alpha";
            }   
            sAlphaS = sP1;
            sAlphaE = sP2;
            ptr += iVParms + 1;
        }
                    
        // Angle
        else if (sVKey == "ANGLE")
        {
            // Ensure 1-2 values
            if (iVParms > 2)
            {
                return "EInvalid Angle";
            }
            if (iVParms == 1)
            {
                sP2 = sP1;
            }
            if ((!is_float(sP1)) || (!is_float(sP2)))
            {
                return "EInvalid Angle";
            }
            sAngleS = sP1;
            sAngleE = sP2;
            ptr += iVParms + 1;
        }
                    
        // Burst
        else if (sVKey == "BURST")
        {
            if (iVParms != 1)
            {
                return "EInvalid Burst";
            }
            if (!is_integer(sP1))
            {
                return "EInvalid Burst";
            }
            sBurst = sP1;
            ptr += 2;
        }
                    
        // Color                      
        else if (sVKey == "COLOR")
        {
            // Ensure 3 or 6 values values
            if ((iVParms != 3) && (iVParms != 6))
            {
                return "EInvalid Color";
            }
            
            if (iVParms == 3)
            {
                sP4 = sP1;
                sP5 = sP2;
                sP6 = sP3;
            }
            if ((!is_float(sP1)) || (!is_float(sP2)) || (!is_float(sP3)) ||
                (!is_float(sP4)) || (!is_float(sP5)) || (!is_float(sP6))) 
            {
                return "EInvalid Color";
            }
            sColorS = "<" + sP1 + "," + sP2 + "," + sP3 + ">";
            sColorE = "<" + sP4 + "," + sP5 + "," + sP6 + ">";
            ptr += iVParms + 1;
        }
                 
        // Glow
        else if (sVKey == "GLOW")
        {
            if (iVParms != 1)
            {
                return "EInvalid Glow";
            }
            sP1 = llToUpper(sP1);
            if ((sP1 != "YES") && (sP1 != "NO"))
            {
                return "EInvalid Glow";
            }
            sGlow = sP1;
            ptr += 2;
        }
                    
        // Omega
        else if (sVKey == "OMEGA")
        {
            // Ensure three values
            if (iVParms != 3)
            {
                return "EInvalid Omega";
            }
            if ((!is_float(sP1)) || (!is_float(sP2)) || (!is_float(sP3)))
            {
                return "EInvalid Omega";
            }
            sOmega = "<" + sP1 + "," + sP2 + "," + sP3 + ">";
            ptr += 4;
        }
            
        // Radius
        else if (sVKey == "RADIUS")
        {
            if (iVParms != 1)
            {
                return "EInvalid Radius";
            }
            if (!is_float(sP1))
            {
                return "EInvalid Radius";
            }
            sRadius = sP1;
            ptr += 2;
        }
                    
        // Speed
        else if (sVKey == "SPEED")
        {
            // Ensure one or two values
            if (iVParms > 2)
            {
                return "EInvalid Speed";
            }
            if (iVParms ==1)
            {
                sP2 = sP1;
            }
            if ((!is_float(sP1)) || (!is_float(sP2)))
            {
                return "EInvalid Speed";
            }
            sSpeedMin = sP1;
            sSpeedMax = sP2;
            ptr += iVParms + 1;
        }
                    
        // Scale                      
        else if (sVKey == "SCALE")
        {
            // Ensure 3 or six values
            if ((iVParms != 3) && (iVParms != 6))
            {
                return "EInvalid Scale";
            }
            if (iVParms == 3)
            {
                sP4 = sP1;
                sP5 = sP2;
                sP6 = sP3;
            }
            if ((!is_float(sP1)) || (!is_float(sP2)) || (!is_float(sP3)) ||
                (!is_float(sP4)) || (!is_float(sP5)) || (!is_float(sP6)))
            {
                return "EInvalid Scale";
            }
            sScaleS = "<" + sP1 + "," + sP2 + "," + sP3 + ">";
            sScaleE = "<" + sP4 + "," + sP5 + "," + sP6 + ">";
            ptr += iVParms + 1;
        }
                 
        // Sleep
        else if (sVKey == "SLEEP")
        {
            if (iVParms != 1)
            {
                return "EInvalid Sleep";
            }
            if (!is_float(sP1))
            {
                return "EInvalid Sleep";
            }
            sSleep = sP1;
            ptr += 2;
        }
                    
        // Texture
        else if (sVKey == "TEXTURE")
        {
            if (iVParms != 1)
            {
                return "EInvalid Texture";
            }
            sTexture = sP1;
            ptr += 2;
        }
        
        // Rotation
        else if (sVKey == "ROTATION")
        {
            if (iVParms != 3)
            {
                return "EInvalid Rotation";
            }
            sRotation = "<" + sP1 + ", " + sP2 + ", " + sP3 + ">";
            ptr += 4;
        }
    }
                
    // Return valid scene string
    return sHeads + "|" + sCDur + "|" + sHDur + "|" + sAccel + "|" + sAge + "|" + sAlphaS + "|" +
           sAlphaE + "|" + sAngleS + "|" + sAngleE + "|" + sBurst + "|" + sColorS + "|" + sColorE +
           "|" + sGlow + "|" + sOmega + "|" + sRadius + "|" + sSpeedMin + "|" + sSpeedMax + "|" +                    sScaleS + "|" + sScaleE + "|" + sSleep + "|" + sTexture + "|" + sRotation;
}

//***********************************************************
say_DB(integer number, string data)        // Display DB Data
//***********************************************************
{
    if (DB)
    {
        llSay(0, "DB" + (string)iDB + "  #" + (string)number + ", " + data);
    }
}      

//************************************************************
default     // Initialize and process ADMINISTRATORS notecard
//************************************************************
{
    state_entry()
    {
        // Reset the nozzles
        COMMAND(LINK_SET, FOUNTAIN_RESET, "", NULL_KEY);
        
        // Create a unique menu channel
        iMenuChan += (integer)llFrand(1000);
        
        // Determine number of notecards
        integer nc_number = llGetInventoryNumber(INVENTORY_NOTECARD);
        integer nc_admin = 0;
        
        // Load notecard names
        integer i; for (i = 0; i < nc_number; i++)
            lNCNames += [llGetInventoryName(INVENTORY_NOTECARD, i)];

        iNCCount = llGetListLength(lNCNames);
        llOwnerSay ((string)iNCCount+" notecards found");

        state load_NC;
    }

}

//**************************************************************
state load_NC          // Process fountain notecard
//**************************************************************
{
    state_entry()
    {
        // Reset lists
        lCommands = [];
        lDOTarget = [];
        lDOSource = [];
        lDOCount = [];
        
        // Get the first line
        iNC = 0;
        llOwnerSay ("Reading notecard "+(string)iActNC);
        kNC = llGetNotecardLine(llList2String(lNCNames, iActNC), iNC);
    }
    
    // Process FOUNTAIN Notecard
    dataserver(key ID, string data)
    {
        // Response from notecard
        if (ID == kNC)
        {
            // End of a lines
            if (data == EOF)
            {
                llOwnerSay ("Done");
               state do_do;
            }
            
            // Ignore comments and blank lines
            data = llStringTrim(data, STRING_TRIM);
            if ((llStringLength(data) > 0) && (llGetSubString(data, 0, 1) != "//"))
            {
                // Process Notecard
                list l = llParseString2List(data, [" "], []);
                string s1 = llToUpper(llList2String(l, 0));
                string s2 = llToUpper(llList2String(l, 1));
                string s3 = llList2String(l, 2);
                
                // Process GROUP line
                if (s1 == "GROUP")
                {
                    // Validate
                    if (llStringLength(s2) > 0)
                    {
                        lCommands += ["G|" + s2];
                    }
                    else
                    {
                        llOwnerSay("Error - Invalid GROUP in " +
                            llList2String(lNCNames, iActNC) + "(" + (string)(iNC + 1)
                            + ")");
                        state error;
                    }
                }
                
                // Process DO line
                else if (s1 == "DO")
                {
                    // Validate
                    if (llStringLength(s2) > 0)
                    {
                        if (llStringLength(s3) > 0)
                        {
                            if (is_integer(s3))
                            {
                                if ((integer)s3 > 0)
                                {
                                    lCommands += ["D|" + s2 + "||" + s3];
                                }
                                else
                                {
                                    llOwnerSay("Error - Invalid DO in " +
                                        llList2String(lNCNames, iActNC) + "(" + (string)(iNC + 1)
                                        + ")");
                                    state error;
                                }
                            }
                            else
                            {
                                lCommands += ["D|" + s2 + "||1"];
                            }
                        }
                        else
                        {
                            lCommands += ["D|" + s2 + "||1"];
                        }     
                    }
                    else
                    {
                        llOwnerSay("Error - Invalid DO in " + 
                            llList2String(lNCNames, iActNC) + "(" + (string)(iNC + 1)
                            + ")");
                        state error;
                    }
                }
                
                // Process Communication Line
                else if ((s1 == "SAY") || (s1 == "MSG"))
                {
                    if (is_integer(s2))
                    {
                        integer i = llSubStringIndex(data, s2) + llStringLength(s2);
                        s3 = llStringTrim(llGetSubString(data, i, -1), STRING_TRIM);
                        if (llStringLength(s3) <= 0)
                        {
                            llOwnerSay("Error - Invalid " + s1 + "in " + 
                                llList2String(lNCNames, iActNC) + "(" + (string)(iNC + 1)
                                + ")");
                            state error;
                        }
                        else
                        {
                            lCommands += [llGetSubString(s1, 0, 0) + "|" + s2 + "|"+ s3];
                        }
                    }
                    else
                    {
                        llOwnerSay("Error - Invalid " + s1 + "in " + 
                            llList2String(lNCNames, iActNC) + "(" + (string)(iNC + 1)
                            + ")");
                        state error;
                    }
                }
                
                // Process Command line
                else
                {
                    s1 = validation(data);
                
                    // Eliminate validation errors
                    if (llGetSubString(s1, 0, 0) == "E")
                    {
                        llOwnerSay("Error - " + llGetSubString(s1, 1, -1) + " in " +
                            llList2String(lNCNames, iActNC) + "(" + (string)(iNC + 1)
                            + ")");
                        state error;
                    }
                
                    // Store Command
                    lCommands += ["C|" + s1];
                }
            }
                    
            // Next Line
            iNC++;
            kNC = llGetNotecardLine(llList2String(lNCNames, iActNC), iNC);
        }
    }
    
    on_rez(integer parm)
    {
        llResetScript();
    }
    
    changed(integer change)
    {
        if (change == CHANGED_INVENTORY)
        {
            llResetScript();
        }
    }
}

//***********************************************
state do_do          // Resolve DOs
//***********************************************
{
    state_entry()
    {
        // Process DO commands
        string entry;
        list l;

        integer doi;
        string docommand;
        string doname;
        string doiterate;

        integer grpi;
        string grpcommand;
        string grpname;

        for (doi = 0; doi < llGetListLength(lCommands); doi++)
        {
            entry = llList2String(lCommands, doi);
            l = llParseString2List(entry, ["|"], []);
            docommand = llList2String(l, 0);
            doname = llList2String(l, 1);
            doiterate = llList2String(l, 2);
            
            if (docommand == "D")
            {
                
                // Scan through commands to find Group
                integer foundDO = FALSE;
                for (grpi = 0; grpi < llGetListLength(lCommands); grpi++)
                {
                    entry = llList2String(lCommands, grpi);
                    l = llParseString2List(entry, ["|"], []);
                    grpcommand = llList2String(l, 0);
                    grpname = llList2String(l, 1);
                    
                    // Did we find a Group header
                    if (grpcommand == "G")
                    {
                        // Does it match the DO name
                        if (doname == grpname)
                        {
                            foundDO = TRUE;
                            // Update the DO pointer
                            lCommands = llListReplaceList(lCommands, ["D|" +
                                doname + "|" + (string)grpi + "|" + doiterate], doi, doi);
                        }
                    }
                }
                if (!foundDO)
                {
                    llOwnerSay("Error - DO " + doname + 
                        " references an unknown GROUP in " +
                        llList2String(lNCNames, iActNC) + "(" + (string)(doi + 1)
                            + ")");
                    state error;
                }
            }
        }
        
        // Done Editing
        state main;
    }
}
                       
                    
//***********************************************
state main          // Main process state
//***********************************************
{
    state_entry()
    {
        // Open Menu Channel
        llListen(iMenuChan, "", NULL_KEY, "");
        
        // Initialize Command Pointer
        iCPtr = 0;
        
        // Start timer if fountain is on
        if (iFountainOn)
        {
            llSetTimerEvent(1.0);
        }
    }
    
    // Menu Initiation
    touch_start(integer num_touched)
    {
        // Create  menu
        kAvKey = llDetectedKey(0);
        string av_name = llKey2Name(kAvKey);
        lMenu = [];
        
        // Validate user
//        if ((llGetOwner() == kAvKey) || 
//            (llListFindList(lAdmin, [llToUpper(av_name)]) >= 0))
        if (TRUE)
        {
            if (iFountainOn)
            {
                lMenu = ["Stop"];
            }
            else
            {
                lMenu = ["Start"];
            }
            lMenu += ["Reset", "List"];
            llDialog(kAvKey, "Select menu option", lMenu, iMenuChan);
        }
    }
    
    // Menu Processing
    listen(integer chan, string name, key ID, string data)
    {
        integer i;
        
        // Validate user
//        if ((llGetOwner() == ID) || 
//            (llListFindList(lAdmin, [llToUpper(name)]) >= 0))
        if (TRUE)
        {
            // Start the Fountain
            if (data == "Start")
            {
                iFountainOn = TRUE;
                llSetTimerEvent(1.0);
                llSay(0, "Fountain Starting");
            }
            
            // Stop the Fountain
            else if (data == "Stop")
            {
                iFountainOn = FALSE;
                llSetTimerEvent(0.0);
                llSay(0, "Fountain Stopping");
            }
            
            // Reset the Fountain
            else if (data == "Reset")
            {
                llSay(0, "Fountain Reset");
                llResetScript();
            }
            
            // Get Notecard List
            else if (data == "List")
            {
                lMenu = [];
                iMenu = 0;
                
                if (iNCCount > 12 )
                {
                    lMenu = ["Next"];
                    for (i = 9; i >= 0; i--)
                    {
                        lMenu += [max_12(lNCNames, i)];
                    }
                }
                else
                {
                    for (i = (iNCCount - 1); i >= 0; i--)
                    {
                        lMenu += [max_12(lNCNames, i)];
                    }
                }
                llDialog(kAvKey, "Select notecard", lMenu, iMenuChan);
            }
            
            // Get the next 10 from list
            else if (data == "Next")
            {
                lMenu = ["Prev"];
                iMenu += 10;
                
                if ((iMenu + 10) >= iNCCount)
                {
                    for (i = (iNCCount - 1); i >= iMenu; i--)
                    {
                        lMenu += [max_12(lNCNames, i)];
                    }
                }
                else
                {
                    lMenu += ["Next"];
                    for (i = (iMenu + 9); i>= iMenu; i--)
                    {
                        lMenu += [max_12(lNCNames, i)];
                    }
                }
                llDialog(kAvKey, "Select notecard", lMenu, iMenuChan);
            }
                
            // Get the previous 10 from the list
            else if (data == "Prev")
            {
                lMenu = [];
                iMenu -= 10;
                if (iMenu > 0)
                {
                    lMenu += ["Prev"];
                }
                lMenu += ["Next"];
                
                for (i = (iMenu + 9); i >= iMenu; i--)
                {
                    lMenu += [max_12(lNCNames, i)];
                }
                llDialog(kAvKey, "Select notecard", lMenu, iMenuChan);
            }            
                    
            // Select the new notecard and process it
            else
            {
                for (i = 0; i < iNCCount; i++)
                {
                    if (max_12(lNCNames, i) == data)
                    {
                        iActNC = i;
                        llSay(0, "Selecting " + llList2String(lNCNames, i) + " notecard");
                        state load_NC;
                    }
                }
            }
        }
    }        
    
    // Process next Command
    timer()
    {
        // Stop other events
        llSetTimerEvent(0.0);
        
        // Wait for a Command to be issued
        integer command_issued = FALSE;
        do
        {
            // Check end of Command List
            if (iCPtr >= llGetListLength(lCommands))
            {
                // Return or go back to start of command list
                if (llGetListLength(lDOTarget) > 0)
                {
                    integer i = llList2Integer(lDOCount, -1);
                    if (i > 0)
                    {
                        iCPtr = llList2Integer(lDOTarget, -1) + 1;
                        i--;
                        lDOCount = llListReplaceList(lDOCount, [(string)i], -1, -1);
                    }
                    else
                    {
                        iCPtr = llList2Integer(lDOSource, -1) +1;
                        lDOTarget = llDeleteSubList(lDOTarget, -1, -1);
                        lDOSource = llDeleteSubList(lDOSource, -1, -1);
                        lDOCount = llDeleteSubList(lDOCount, -1, -1);
                    }
                }
                else
                {
                    iCPtr = 0;
                }
            }
            
            // Get entry from Command Table
            list l = llParseString2List(llList2String(lCommands, iCPtr), ["|"], []);
            string s1 = llList2String(l, 0);
            string s2 = llList2String(l, 1);
            string s3 = llList2String(l, 2);
            string s4 = llList2String(l, 3);
        
            // Handle GROUP entry
            if (s1 == "G")
            {
                // Return or go back to start of command list
                if (llGetListLength(lDOTarget) > 0)
                {
                    integer i = llList2Integer(lDOCount, -1);
                    if (i > 0)
                    {
                        iCPtr = llList2Integer(lDOTarget, -1) + 1;
                        i--;
                        lDOCount = llListReplaceList(lDOCount, [(string)i], -1, -1);
                    }
                    else
                    {
                        iCPtr = llList2Integer(lDOSource, -1) +1;
                        lDOTarget = llDeleteSubList(lDOTarget, -1, -1);
                        lDOSource = llDeleteSubList(lDOSource, -1, -1);
                        lDOCount = llDeleteSubList(lDOCount, -1, -1);
                    }
                }
                else
                {
                    iCPtr = 0;
                }
            }
            
            // Handle DO entry
            else if (s1 == "D")
            {
                // Debug
                if (DB)
                {
                    llSay(0, "Debug: (" + (string)iDB + ") " + llList2String(lCommands, iCPtr));
                    iDB++;
                }
            
                // Save Stack entries
                lDOTarget += [s3];
                lDOSource += (string)iCPtr;
                s4 = (string)((integer)s4 -1);
                lDOCount += [s4];
               
                // Go to new Group
                iCPtr = (integer)s3 + 1;
                
                // Check for recursive
                if (llGetListLength(lDOTarget) > 10)
                {
                    llOwnerSay("Error - Recursive use of DO");
                    state error;
                }
            }
            
            // Handle SAY entry
            else if (s1 == "S")
            {
                if (DB)
                {
                    llSay(0, "Debug: (" + (string)iDB + ") " + llList2String(lCommands, iCPtr));
                    iDB++;
                }

                llSay((integer)s2, s3);
                iCPtr++;
            }
            
            // Handle MSG entry
            else if (s1 == "M")
            {
                if (DB)
                {
                    llSay(0, "Debug: (" + (string)iDB + ") " + llList2String(lCommands, iCPtr));
                    iDB++;
                }
                
                COMMAND(LINK_SET, (integer)s2, s3, NULL_KEY);
                iCPtr++;
            }
            
            // Handle Command to Heads
            else if (s1 == "C")
            {
                // Debug
                if (DB)
                {
                    llSay(0, "Debug: (" + (string)iDB + ") " + llList2String(lCommands, iCPtr));
                    iDB++;
                }
            
                string s = llGetSubString(llList2String(lCommands, iCPtr), 2, -1);        
                COMMAND(LINK_SET, FOUNTAIN_MSG, s, NULL_KEY);
        
                // Save Duration of Command
                list l1 = llParseString2List(s, ["|"], []);
                float dur = llList2Float(l1, 1);
        
                // Set up next Command
                iCPtr++;
                
                // Break out of Command loop
                command_issued = TRUE;
                
                // Set Timer for next action
                if (dur <= 0.0)
                {
                    dur = 0.1;
                }
                llSetTimerEvent(dur);
            }
        } 
        while (!command_issued);
    }
                                                        
    on_rez(integer parm)
    {
        llResetScript();
    }        

    changed(integer change)
    {
        if (change == CHANGED_INVENTORY)
        {
            llResetScript();
        }
    }

}
        
//***********************************************
state error          // Notecard Error Handler
//***********************************************
{
    on_rez(integer parm)
    {
        llResetScript();
    }        

    changed(integer change)
    {
        if (change == CHANGED_INVENTORY)
        {
            llResetScript();
        }
    }
}        